#!/bin/bash
alertTitleMigration="Device Management Migration"

alertTitlePasswordError="Password Error Notification"

alertTitleManualEnrollment="Enrollment Notification"
alertMessageManualEnrollment="Your device is currently unenrolled.\n As per our security guidelines, please navigate to System Settings and search for Profiles and install the downloaded enrollment payload."

enableLocalLogFile=0
LOGFOLDER="/Library/Logs/MDM-Migration"
LOG=$LOGFOLDER"/MDM-Migration.log"

bearerToken=""
JSS=""
ADEResponse=""
enrollmentTypeFlag=""
AdminPromoted=false
LogoPath=""

getBearerToken() {
	response=$(curl -s -u "matech":"D31n4aD3S0!" https://marinacademy.jamfcloud.com/api/v1/auth/token -X POST)
	bearerToken=$(echo "$response" | plutil -extract token raw -)
	if [ -z "$bearerToken" ] || [ "$(echo $bearerToken | grep 'Could not extract value' )" ]; then
		logme "[ERROR] Failed to retrieve Bearer Token. Exiting..."
		demoteUserToStandard
		exit 1;
	else
		echo "Bearer Token: $bearerToken"
	fi
}

getJSSID() {
	DeviceSerialNumber=$(system_profiler SPHardwareDataType | awk '/Serial/ {print $4}')
	logme "Device Serial Number is--> $DeviceSerialNumber"
	JSS=$(curl -H "Accept: text/xml" -H "Authorization: Bearer ${bearerToken}" "https://marinacademy.jamfcloud.com/JSSResource/computers/serialnumber/${DeviceSerialNumber}/subset/general" | /usr/bin/perl -lne 'BEGIN{undef $/} while (/<id>(.*?)<\/id>/sg){print $1}' | head -n 1)
	if [ -z "$JSS" ]; then
		logme "[ERROR] Failed to retrieve JSS ID. Exiting..."
		demoteUserToStandard
		exit 1;
	else
		echo "Device JSS ID is: $JSS"
	fi
}

getADEResponse() {
	logme "Querying the local system for enrollment profiles..."
	HostResponse="$(profiles show -type enrollment 2>&1)"
	echo "$HostResponse"
	if [ "$(echo $HostResponse | grep '(null)\|Client is not DEP enabled' )" ]; then
		logme "No ADE profile associated. Setting Safari Enrollment Flag..."
	    if [ -z "https://enroll.mosyle.com/?account=maorg" ]; then
			logme "[ERROR] Missing URL for Safari enrollment. Exiting..."
			demoteUserToStandard
			exit 1;
	    fi
		enrollmentTypeFlag=0
	elif [ "$(echo $HostResponse | grep 'ConfigurationURL' )" ]; then
		logme "ADE profile associated. Setting ADE Enrollment Flag..."
		enrollmentTypeFlag=1
	else
		logme "[ERROR] $HostResponse"
		demoteUserToStandard
		exit 1;
	fi
}

promptManualEnrollment() {
	/usr/bin/osascript -e "display alert \"$alertTitleManualEnrollment\" message \"$alertMessageManualEnrollment\" as critical buttons {\"Open System Settings\"}"
	open x-apple.systempreferences:com.apple.Profiles-Settings.extension
}

verifyPasswordForUser() {
    VERIFYPASSWORD=$(dscl /Local/Default -authonly "${CURRENTCONSOLEUSER}" "${PASSWORDINPUT}")
    if [ -z "$VERIFYPASSWORD" ]; then
      return 0
    else
      return 1
    fi
}

readInPasswordForUser() {
	n=1
	if [ -f $LogoPath ]; then
		PASSWORDINPUT="$(osascript -e 'display dialog "Please enter the password for '${CURRENTCONSOLEUSER}':" default answer ""  with icon POSIX file "'${LogoPath}'" buttons {"Confirm"} default button 1 with hidden answer' -e 'text returned of result' 2>/dev/null)"
	else
		PASSWORDINPUT="$(osascript -e 'display dialog "Please enter the password for '${CURRENTCONSOLEUSER}':" default answer "" buttons {"Confirm"} default button 1 with hidden answer' -e 'text returned of result' 2>/dev/null)"
    fi
	if ! verifyPasswordForUser; then
		while [ $n -lt 10 ]; do
			logme "End-User provided incorrect password. ($n of 10)"
			if [ -f $LogoPath ]; then
				PASSWORDINPUT="$(osascript -e 'display dialog "Incorrect Password.\nPlease enter the correct password for '${CURRENTCONSOLEUSER}':" default answer "" with icon POSIX file "'${LogoPath}'" buttons {"Confirm"} default button 1 with hidden answer' -e 'text returned of result' 2>/dev/null)"
			else
				PASSWORDINPUT="$(osascript -e 'display dialog "Incorrect Password.\nPlease enter the correct password for '${CURRENTCONSOLEUSER}':" default answer "" buttons {"Confirm"} default button 1 with hidden answer' -e 'text returned of result' 2>/dev/null)"
			fi

			if verifyPasswordForUser; then
				logme "Correct password provided. Continuing..."
				return 0
			fi
			n=$((n + 1)) 
		done
		logme "[ERROR] Too many incorrect password attempts ($n of 10). Exiting..."
		if [ -f $LogoPath ]; then
			/usr/bin/osascript -e "display dialog \"Wrong password try again\" with title \"$alertTitlePasswordError\"  with icon POSIX file \"$LogoPath\" buttons {\"Exit\"}"
		else
			/usr/bin/osascript -e "display alert \"$alertTitlePasswordError\" message \"Wrong password try again\" as critical buttons {\"Exit\"}"
		fi
		demoteUserToStandard
		exit 1
    else
		logme "Correct password provided. Continuing..."
      	return 0
    fi
}

logme()
{
	if [[ $enableLocalLogFile -eq 1 ]]; then
		if [ ! -d "$LOGFOLDER" ]; then
			mkdir -p $LOGFOLDER
		fi
		if [ -z "$1" ]; then
			echo $( date )" - logme function call error: no text passed to function! Please recheck code!"
		fi
		echo -e "[$(date "+%b %d, %Y %H:%M:%S")] $1" 2>&1 | tee -a $LOG
	else
		echo "$1"
	fi
}

promoteUserToAdmin()
{
	logme "Promoting current user to Admin..."
	dseditgroup -o edit -a "${CURRENTCONSOLEUSER}" -u ${CURRENTCONSOLEUSER} -P ${PASSWORDINPUT} -t user "admin"; 
	AdminPromoted=true
}

demoteUserToStandard()
{
	if [ $AdminPromoted == true ]; then
		logme "Demoting current user to Standard..."
		dseditgroup -o edit -d "${CURRENTCONSOLEUSER}" -u ${CURRENTCONSOLEUSER} -P ${PASSWORDINPUT} -t user "admin";
	fi
}

downloadLogo()
{
	curl -o /tmp/logo-maorg-macosappicon.png https://mosylemanagerweb.blob.core.windows.net/maorg-public/logo-maorg-macosappicon.png?v=1711148597  
	LogoPath="/tmp/logo-maorg-macosappicon.png"                
}

removeLogo()
{
	if [ -f $LogoPath ]; then
		rm $LogoPath;
	fi
}

logme "Starting Migration to new MDM"

logme "Getting Bearer Token"
getBearerToken

logme "Getting JSS ID"
getJSSID

logme "Checking to make sure someone is logged in and if so, get their username"
CURRENTCONSOLEUSER=$(stat -f "%Su" /dev/console);
if [ "$CURRENTCONSOLEUSER" == root ]; then
	logme "ERROR! No user logged in. Unable to migrate device. Please wait for an end-user and attempt again..."
	demoteUserToStandard
	exit;
fi

downloadLogo

logme "Notifying the user that this migration is taking place."
if [ -f $LogoPath ]; then
	/usr/bin/osascript -e "display dialog \"Hello ${CURRENTCONSOLEUSER}.\n\nMigration in Process\" with title \"$alertTitleMigration\"  with icon POSIX file \"$LogoPath\" buttons {\"Continue\"}"
else
	/usr/bin/osascript -e "display alert \"$alertTitleMigration\" message \"Hello ${CURRENTCONSOLEUSER}.\n\nMigration in Process\" as critical buttons {\"Continue\"}"
fi

logme "Prompting user for credentials so that the migration can be done."
readInPasswordForUser

if groups $CURRENTCONSOLEUSER | grep -q -w admin; then 
	logme "$CURRENTCONSOLEUSER is a local admin. Continuing with migration..."
else 
	logme "[ERROR] $CURRENTCONSOLEUSER is not an admin. Promoting user to admin..."
	promoteUserToAdmin
fi

logme "Determining if this device is in ASM/ABM (ADE)"
getADEResponse

logme "Sending UnmanageDevice API request to Jamf..."
curl -X POST -H "Accept: application/xml" -H "Authorization: Bearer ${bearerToken}" "https://marinacademy.jamfcloud.com/JSSResource/computercommands/command/UnmanageDevice/id/${JSS}"

sleep 10

WaitingCount=0
GotProfiles="Placeholder"

logme "Waiting for profiles to be removed."
until [ -z "$GotProfiles" ]; do
	profiles sync
	GotProfiles=$(profiles list -all | grep com.jamf.)
	logme "Jamf Profiles are still here...  WAITING!"
	sleep 5
	let WaitingCount=WaitingCount+1
	
	if [ "$WaitingCount" -ge 36 ]; then
		logme "[ERROR] We've waited three minutes for these profiles to disable. Verify if the 'Unmanage' command from Jamf has been delivered to the device."
		demoteUserToStandard
		exit 1
	fi
done

computerInfo="<computer><general><remote_management><managed>false</managed></remote_management></general></computer>"
echo "Setting an attribute in Jamf so that we can use a smart group to determine all those machines that have migrated for further deletion"
curl -k 0 "https://marinacademy.jamfcloud.com/JSSResource/computers/serialnumber/$DeviceSerialNumber" -H "Authorization: Bearer ${bearerToken}" -H "Content-Type: text/xml" -X PUT -d "$computerInfo"

logme "Removing MDM profile..."
jamf removeMdmProfile

sleep 5
logme "Removing any other profiles left behind..."
SkipUsers='Shared\|_\|nobody\|root\|daemon'
for username in $(dscl . list /Users | grep -v $SkipUsers); do
	identifier="$(/usr/bin/profiles -L -U $username | awk "/attribute/" | awk '{print $4}')"
	echo "Removing profile: $identifier"
	/usr/bin/profiles -R -p "$identifier" -U $username
done

profileList="$(profiles -P | grep 'profileIdentifier:' | awk '{print $4}')"
IFS=$'\n'
count=0
for PID in $profileList; do
	profiles -R -p "$PID" > /dev/null 2> /dev/null
done
sleep 5
if [ ! -z "" ]; then
	logme "Reconnecting to wifi if wifi was controlled by a profile..."
	WIFIEN="$(networksetup -listallhardwareports | awk '/Wi-Fi/{getline; print $2}')"
	networksetup -setairportnetwork "$WIFIEN" "" ""
	sleep 10
	networksetup -setairportpower "$WIFIEN" off
	sleep 2.5
	networksetup -setairportpower "$WIFIEN" on
	sleep 10
fi

logme "Removing JAMF Framework..."
jamf removeFramework

logme "Beginning enrollment into new MDM"
if [[ $enrollmentTypeFlag -eq 0 && 0 -eq 1 ]]; then
	logme "No ADE profile associated. Proceeding with Safari (URL) enrollment..."
	open -a Safari "https://enroll.mosyle.com/?account=maorg"
	promptManualEnrollment;
	
	logme "Waiting for user to enroll device..."
	sleep 60
	profileList="$(profiles -P | grep 'profileIdentifier:' | awk '{print $4}')"
	until [ ! -z "$profileList" ]; do
		promptManualEnrollment;
		logme "Waiting for user to enroll device..."
		sleep 60
		profileList="$(profiles -P | grep 'profileIdentifier:' | awk '{print $4}')"
	done
	logme "Profile detected. Exiting..."
	
elif [ $enrollmentTypeFlag -eq 1 ]; then
	logme "ADE profile associated. Proceeding with ADE enrollment via Terminal..."
	if [ "$( profiles renew -type enrollment 2>&1 | grep 'DEP enrollment failed' )" ]; then
		/usr/bin/su -l $CURRENTCONSOLEUSER -c "echo $PASSWORDINPUT | sudo -S profiles renew -type enrollment"
	fi
	
	logme "Waiting for user to enroll device..."
	sleep 60
	profileList="$(profiles -P | grep 'profileIdentifier:' | awk '{print $4}')"
	until [ ! -z "$profileList" ]; do
		if [ "$( profiles renew -type enrollment 2>&1 | grep 'DEP enrollment failed' )" ]; then
			/usr/bin/su -l $CURRENTCONSOLEUSER -c "echo $PASSWORDINPUT | sudo -S profiles renew -type enrollment"
		fi
		logme "Waiting for user to enroll device..."
		sleep 60
		profileList="$(profiles -P | grep 'profileIdentifier:' | awk '{print $4}')"
	done
	logme "Profile detected. Exiting..."
		
	
else
	logme "No ADE profile associated. Skipping device enrollment... "
fi
demoteUserToStandard
removeLogo
logme "Migration Completed Successfully!"
